/*
 Copyright (c) 2000 - 2005, The JAP-Team
 All rights reserved.
 Redistribution and use in source and binary forms, with or without modification,
 are permitted provided that the following conditions are met:

  - Redistributions of source code must retain the above copyright notice,
 this list of conditions and the following disclaimer.

  - Redistributions in binary form must reproduce the above copyright notice,
 this list of conditions and the following disclaimer in the documentation and/or
 other materials provided with the distribution.

  - Neither the name of the University of Technology Dresden, Germany nor the names of its contributors
 may be used to endorse or promote products derived from this software without specific
 prior written permission.


 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS'' AND ANY EXPRESS
 OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS
 BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
 IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 */
/* Hint: This file may be only a copy of the original file which is always in the JAP source tree!
 * If you change something - do not forget to add the changes also to the JAP source tree!
 */
package anon.infoservice;

/**
 * This is a generic definition for a database entry. Every database entry must implement this
 * methods.
 */
public abstract class AbstractDatabaseEntry
{
	public static final String XML_LAST_UPDATE = "LastUpdate";
	public static final String XML_ATTR_LAST_UPDATE = "lastUpdate";

	/**
	 * Stores the time when this entry will be deleted from the database.
	 */
	private long m_expireTime;

	private long m_creationTime;

	/**
	 * Creates a new AbstractDatabaseEntry with the specified expire time.
	 *
	 * @param a_expireTime The time when this entry will be deleted from the database.
	 */
	public AbstractDatabaseEntry(long a_expireTime)
	{
		m_expireTime = a_expireTime;
		m_creationTime = System.currentTimeMillis();
	}

	/**
	 * Returns if this entry is newer than the given entry. If the entry that already is in the database
	 * is newer than an entry that is added to the db, nothing is changed and the old entry remains in
	 * the db.
	 * @param a_oldEntry AbstractDatabaseEntry
	 * @return if this entry is newer than the given entry, or if the given entry is null
	 */
	public final boolean isNewerThan(AbstractDatabaseEntry a_oldEntry)
	{
		return a_oldEntry == null || (getVersionNumber() > a_oldEntry.getVersionNumber() ||
				(getLastUpdate() > a_oldEntry.getLastUpdate() &&
				 getVersionNumber() == a_oldEntry.getVersionNumber()));
	}

	/**
	 * Returns whether this entry was generated by a user within the JAP client (true) or
	 * was generated by an InfoService itself (false).
	 *
	 * @return Whether this entry is user-defined.
	 */
	public boolean isUserDefined()
	{
		return false;
	}

	public boolean isPersistanceDeletionAllowed()
	{
		return false;
	}
	
	public void deletePersistence()
	{
		// do nothing
	}

	/**
	 * Returns a unique ID for a database entry.
	 *
	 * @return The ID of this database entry.
	 */
	public abstract String getId();

	/**
	 * Returns the time (see System.currentTimeMillis()) when this DatabaseEntry will be removed
	 * from the Database, if it is not updated meanwhile.
	 *
	 * @return The expire time for this DatabaseEntry.
	 */
	public final long getExpireTime()
	{
		return m_expireTime;
	}

	/**
	 * Returns the time in milliseconds when this db entry was created from the origin instance.
	 * Never (!) insert the local creation time here, as this will DoS the InfoServices in an endless loop.
	 * @return the time in milliseconds when this db entry was created from the origin instance
	 */
	public abstract long getLastUpdate();

	/**
	 * Returns the time in milliseconds when this db entry was create, that means read from the network or
	 * created from internal values.
	 * @return the time in milliseconds when this db entry was created
	 */
	public final long getCreationTime()
	{
		return m_creationTime;
	}

	public final void resetCreationTime()
	{
		m_creationTime = System.currentTimeMillis();
	}


	/**
	 * Returns version number which is used to determine the more recent infoservice entry, if two
	 * entries are compared (higher version number -> more recent entry).
	 * Never (!) insert the local creation time here, as this will DoS the InfoServices in an endless loop.
	 *
	 * @return The version number for this entry.
	 */
	public abstract long getVersionNumber();

}
